// GDesk InfoView plug-in - part of the GDesk suite
//
// Copyright (C) 2007-8 Gary Leach
// 
// See Interface.h for description of member functions

// This is the source code for the InfoView plug-in for GDesk.
// It is presented in order to give plug-in creators a good grounding in how plug-ins work for GDesk.
// InfoView was always partly designed as tutorial code but it is by no means an example of great coding style!

// As well as changing this code and the UIDs you'll need to change the other files too
// It's particularly important to change the names (Display_Name and Default_Data) in the RSS file otherwise GDesk will get confused!

// If you create new plug-ins, or alter this one, you are encouraged to share your source code as well as your compiled plug-in.
// Post your creations to gdesk.wetpaint.com

#include <e32std.h>
#include <ImplementationProxy.h>
#include "Interface.h"

#include <Hal.h>
#include <ASCliSession.h>
#include <ASShdAlarm.h>


// The Dll and Implementation UIDs are specific to your plug-in
// Please don't use the values defined for InfoView, this will lead to enormous problems
// It's easy and free to apply for UIDs - see www.symbiansigned.com for details.

// ECOM UID 0x10009D8D (global)
// I/f UID 0xA0002181 (global to GDesk)
// DLL UID XXXXXXXX (per plug-in)
// Imp UID XXXXXXXY (per plug-in)

#define DllUID 0xXXXXXXXX	// get some UIDs from Symbian!
#define ImplementationUID 0xXXXXXXXY

#include <Etel3rdParty.h>

static const CTelephony::TNotificationEvent  sEvents[] = {
	CTelephony::EIndicatorChange,
	CTelephony::EBatteryInfoChange,
	CTelephony::ESignalStrengthChange,
	CTelephony::ECurrentNetworkInfoChange,
};
static const CTelephony::TCancellationRequest  sCancellations[] = {
	CTelephony::EGetIndicatorCancel, CTelephony::EIndicatorChangeCancel,
	CTelephony::EGetBatteryInfoCancel, CTelephony::EBatteryInfoChangeCancel,
	CTelephony::EGetSignalStrengthCancel, CTelephony::ESignalStrengthChangeCancel,
	//CTelephony::EGetCurrentNetworkInfoCancel , CTelephony::ECurrentNetworkInfoChangeCancel,
	CTelephony::EGetCurrentNetworkNameCancel, CTelephony::ECurrentNetworkInfoChangeCancel,
};

class CNotifier : public CActive
{
public:
	CNotifier( int nType );
	~CNotifier() { Cancel(); if (iTelephony != NULL)  delete iTelephony; iTelephony = NULL; }
	//void ShutDown() { Cancel(); }
	int GetError() { return iError; }
	void ClearError() { iError = KErrNone; }
	int GetUpdated() { int nUpdated = iUpdated; iUpdated = 0; return nUpdated; }
	int GetIndicators() { if (iValid == 0) return -1; return pcIndicatorData->iIndicator; }
	int GetBatteryLevel() { if (iValid == 0) return -1; return pcBatteryData->iChargeLevel; }
	int GetSignalLevel() { if (iValid == 0) return -1; return pcSignalData->iBar | (pcSignalData->iSignalStrength*16); }//SignalStrength; }
	TDesC& GetNetworkName() { if (iDataType == 0) return pcNetworkName->iNetworkName; return pcNetworkData->iLongName; }
private:
	void RunL();
	void DoCancel();
private:
	CTelephony* iTelephony;
	int iUpdated;
	int iType;
	int iValid;
	int iFirstTime;
	int iDataType;
	int iError;
	TRequestStatus iReqStatus;
	// CTelephony::TNotificationEvent lists the classes associated with each event.
	CTelephony::TNetworkInfoV1 cNetworkData;	// ECurrentNetworkInfoChange, ECurrentNetworkInfoChangeCancel
	CTelephony::TNetworkInfoV1 cNetworkDataStore;
	//CTelephony::TCallStatusV1 iLineStatus;
	//CTelephony::TCallStatusV1Pckg iLineStatusPckg;
	CTelephony::TNetworkInfoV1Pckg cNetworkPkg;

	CTelephony::TIndicatorV1* pcIndicatorData;		// EIndicatorChange, EIndicatorChangeCancel
	CTelephony::TBatteryInfoV1* pcBatteryData;		// EBatteryInfoChange, EBatteryInfoChangeCancel
	CTelephony::TSignalStrengthV1* pcSignalData;	// ESignalStrengthChange, ESignalStrengthChangeCancel
	CTelephony::TNetworkInfoV1* pcNetworkData;
	CTelephony::TNetworkNameV1* pcNetworkName;
};
CNotifier::CNotifier( int nType ) : CActive( EPriorityStandard ), cNetworkPkg(cNetworkData)
{
#if defined (__WINS__)
	int nErr = -1;
#else
	TRAPD(nErr, iTelephony = CTelephony::NewL());
#endif
	if (nErr != KErrNone)
	{
		iError = nErr;
		return;
	}

	CActiveScheduler::Add(this);

	iType = nType;
	iValid = 0;
	iFirstTime = 1;
	iUpdated = 0;
	iError = KErrNone;
	iDataType = 0;

	pcIndicatorData = (CTelephony::TIndicatorV1*)&cNetworkDataStore;
	pcBatteryData = (CTelephony::TBatteryInfoV1*)&cNetworkDataStore;
	pcSignalData = (CTelephony::TSignalStrengthV1*)&cNetworkDataStore;
	pcNetworkData = &cNetworkDataStore;
	pcNetworkName = (CTelephony::TNetworkNameV1*)&cNetworkDataStore;

	switch (nType)
	{
	case 0: iTelephony->GetIndicator(iStatus, cNetworkPkg); break;
	case 1: iTelephony->GetBatteryInfo(iStatus, cNetworkPkg); break;
	case 2: iTelephony->GetSignalStrength(iStatus, cNetworkPkg); break;
	case 3: iTelephony->GetCurrentNetworkName(iStatus, cNetworkPkg); break;
	//case 3: iTelephony->GetCurrentNetworkInfo(iStatus, cNetworkPkg); break;
	default: iTelephony->NotifyChange(iStatus, sEvents[iType], cNetworkPkg); break;
	}
	SetActive();
}
void CNotifier::RunL()
{
	if( iStatus==KErrNone )
	{
		Mem::Copy(&cNetworkDataStore, &cNetworkData, sizeof(cNetworkDataStore));
		iValid = 1;
		iUpdated = 1;
		iDataType = iFirstTime ? 0 : 1;
	}
	else
	{
		iError = iStatus.Int();
	}
	iFirstTime = 0;
	iTelephony->NotifyChange( iStatus, sEvents[iType], cNetworkPkg );
	SetActive();
}
void CNotifier::DoCancel()
{
	iTelephony->CancelAsync( sCancellations[iType*2+1-iFirstTime] );
}

/////////////////

class CGDeskPluginInfoView : public CGDeskInterface
{
public:
	static CGDeskPluginInfoView* NewL();

	int Register(const char* pbKeyData, const TBuf16<30>& cIMEI);
	int Activate(void);
	int Update(int nInactivityTicks);
	int PreDraw(int nWhich, void* pvDataIn, int* pnNextTypeOut, int* pnNextParameterOut);
	void Draw(const TPoint& cPoint, CFbsBitGc& gc, int nHighlightRgb);
	const TSize& Size(void) { return m_cSize; }
	int GetOptionsMenuItem(int nIndex, TPtrC& cString, int& nFlags);
	void SelectOptionsMenuItem(int nIndex, int nValue = 0);
	int Control(int nMX, int nMY, int nMButton, int nKeyX, int nKeyY, int nKeyZ, int nKeyFire);
	int GetWriteSize(void) { return sizeof(m_nScale) + sizeof(m_nFlags) + sizeof(m_nInfoId) + sizeof(m_nBackgroundRgb) + sizeof(m_nTextRgb); }
	void Write(char* pbData)
	{
		*(int*)pbData = m_nScale; pbData += sizeof(m_nScale);
		*(int*)pbData = m_nInfoId; pbData += sizeof(m_nInfoId);
		*(int*)pbData = m_nFlags; pbData += sizeof(m_nFlags);
		*(int*)pbData = m_nBackgroundRgb.Value(); pbData += sizeof(int);
		*(int*)pbData = m_nTextRgb.Value(); pbData += sizeof(int);
	}
	void Read(char* pbData, int nBytes)
	{
		m_nScale = *(int*)pbData; pbData += sizeof(m_nScale);
		int nInfoId = *(int*)pbData; pbData += sizeof(m_nInfoId);
		m_nFlags = *(int*)pbData; pbData += sizeof(m_nFlags);
		m_nBackgroundRgb = *(int*)pbData; pbData += sizeof(int);
		m_nTextRgb = *(int*)pbData; pbData += sizeof(int);
		m_nBackgroundRgb &= 0xFFFFFF;
		m_nTextRgb &= 0xFFFFFF;
		SetSize();
		SetInfoId(nInfoId);
	}
	void SetSize(void);

	int InterfaceVersion(void) { return c_nInterfaceVersion; }
	int PluginVersion(void) { return 0x00010000; }

	void SetInfoId(int nId);
private:
	CGDeskPluginInfoView();
	~CGDeskPluginInfoView();
	void ConstructL();

	HBufC** m_ppcStrings;
	HBufC& GetString(int n) { if (n < 0 || n > *(int*)&m_ppcStrings[1]) n = 1; return *(m_ppcStrings[n+2-1]); }

	int m_nScale;
	int m_nInfoId;
	int m_nFlags;
	TRgb m_nBackgroundRgb, m_nTextRgb;
	//
	TSize m_cSize;
	CFbsFont* m_pcFont;
	//
	int m_nLastInfo;
	TBuf<32> m_cInfoString;
	CNotifier* m_pcNotifier;
};

CGDeskPluginInfoView* CGDeskPluginInfoView::NewL()
{
	CGDeskPluginInfoView* self=new(ELeave) CGDeskPluginInfoView();
	CleanupStack::PushL(self);
	self->ConstructL();
	CleanupStack::Pop(self);
	return self;
}

CGDeskPluginInfoView::~CGDeskPluginInfoView()
{
}

CGDeskPluginInfoView::CGDeskPluginInfoView()
{
}

void CGDeskPluginInfoView::ConstructL()
{
	m_pcNotifier = NULL;

	m_nScale = 1;
	m_nInfoId = -1;
	m_nFlags = 0;
	SetSize();
	SetInfoId(0);	// show free RAM

	m_nTextRgb = (TRgb)0xE0FFFF;
	m_nBackgroundRgb = (TRgb)0x000000;
}

int CGDeskPluginInfoView::Register(const char* pbKeyData, const TBuf16<30>& cIMEI)
{
	return 0;
}

void CGDeskPluginInfoView::SetInfoId(int nId)
{
	if (m_pcNotifier != NULL) { m_pcNotifier->Cancel(); delete m_pcNotifier; m_pcNotifier = NULL; }

	m_nLastInfo = -1;
	switch (nId)
	{
	case 1: m_pcNotifier = new CNotifier(1); break;
	case 2: m_pcNotifier = new CNotifier(2); break;
	case 3: m_pcNotifier = new CNotifier(0); break;
	//case 4: m_pcNotifier = new CNotifier(3); break;
	}
	m_nInfoId = nId;
}

void CGDeskPluginInfoView::SetSize(void)
{
	int nFont = 9 + m_nScale * 2;
	m_cSize.iWidth = nFont * 6;
	m_cSize.iHeight = nFont + 3;
}

#define R_TYPE 1
#define R_SIZE 2
#define R_BACKGROUND_RGB 3
#define R_TEXT_RGB 4
#define R_TEXT_OUTLINE 5
#define R_DISPLAY_IN_MB 6
#define R_SHOW_CHARGING 7
#define R_SHOW_MESSAGE 8
#define R_SHOW_NETWORK_AVAIL 9
#define R_SHOW_PHONECALL 10
#define R_FREE_RAM 11
#define R_BATTERY 12
#define R_SIGNAL 13
#define R_INDICATORS 14
#define R_NEXT_ALARM 15
#define R_SMALL 16
#define R_SMALL_MEDIUM 17
#define R_MEDIUM 18
#define R_MEDIUM_LARGE 19
#define R_LARGE 20
#define R_NO_MESSAGE 21
#define R_C 22
#define R_N 23
#define R_P 24

int CGDeskPluginInfoView::GetOptionsMenuItem(int nIndex, TPtrC& cString, int& nFlags)
{
	switch (nIndex)
	{
	case 0: cString.Set(GetString(R_TYPE)); nFlags = 4 | 2; return 1;
	case 1: cString.Set(GetString(R_SIZE)); nFlags = 0 | 2; return 1;
	case 2: cString.Set(GetString(R_BACKGROUND_RGB)); nFlags = 0x10 | (m_nBackgroundRgb.Value() << 8); return 1;
	case 3: cString.Set(GetString(R_TEXT_RGB)); nFlags = 0x10 | (m_nTextRgb.Value() << 8); return 1;
	case 4: cString.Set(GetString(R_TEXT_OUTLINE)); nFlags = ((m_nFlags & 0x1) ? 0 : 1); return 1;
	case 5:
		if (m_nInfoId == 0) { cString.Set(GetString(R_DISPLAY_IN_MB)); nFlags = ((m_nFlags & 0x2) ? 1 : 0); return 1; }
		if (m_nInfoId == 3) { cString.Set(GetString(R_SHOW_CHARGING)); nFlags = (!(m_nFlags & 0x2) ? 1 : 0); return 1; }
		if (m_nInfoId == 4) { cString.Set(GetString(R_SHOW_MESSAGE)); nFlags = ((m_nFlags & 0x2) ? 1 : 0); return 1; }
		return 0;
	case 6:
		if (m_nInfoId == 3) { cString.Set(GetString(R_SHOW_NETWORK_AVAIL)); nFlags = (!(m_nFlags & 0x4) ? 1 : 0); return 1; }
		return 0;
	case 7:
		if (m_nInfoId == 3) { cString.Set(GetString(R_SHOW_PHONECALL)); nFlags = (!(m_nFlags & 0x8) ? 1 : 0); return 1; }
		return 0;

	case 1*100+0: cString.Set(GetString(R_FREE_RAM)); nFlags = (m_nInfoId == 0) ? 1 : 0; return 1;
	case 1*100+1: cString.Set(GetString(R_BATTERY)); nFlags = (m_nInfoId == 1) ? 1 : 0; return 1;
	case 1*100+2: cString.Set(GetString(R_SIGNAL)); nFlags = (m_nInfoId == 2) ? 1 : 0; return 1;
	case 1*100+3: cString.Set(GetString(R_INDICATORS)); nFlags = (m_nInfoId == 3) ? 1 : 0; return 1;
	case 1*100+4: cString.Set(GetString(R_NEXT_ALARM)); nFlags = (m_nInfoId == 4) ? 1 : 0; return 1;
	//case 1*100+4: cString.Set(_L("Network")); nFlags = (m_nInfoId == 4) ? 1 : 0; return 1;

	case 2*100+0: cString.Set(GetString(R_SMALL)); nFlags = 4 | ((m_nScale == 0) ? 1 : 0); return 1;
	case 2*100+1: cString.Set(GetString(R_SMALL_MEDIUM)); nFlags = (m_nScale == 1) ? 1 : 0; return 1;
	case 2*100+2: cString.Set(GetString(R_MEDIUM)); nFlags = (m_nScale == 2) ? 1 : 0; return 1;
	case 2*100+3: cString.Set(GetString(R_MEDIUM_LARGE)); nFlags = (m_nScale == 3) ? 1 : 0; return 1;
	case 2*100+4: cString.Set(GetString(R_LARGE)); nFlags = (m_nScale == 4) ? 1 : 0; return 1;
	}
	return 0;
}

void CGDeskPluginInfoView::SelectOptionsMenuItem(int nIndex, int nValue)
{
	int nMenu = nIndex / 100;
	int nItem = nIndex - (nMenu * 100);
	switch (nMenu)
	{
	case 0:
		if (nItem == 2) m_nBackgroundRgb = (TRgb)nValue;
		else if (nItem == 3) m_nTextRgb = (TRgb)nValue;
		else if (nItem == 4) m_nFlags ^= 0x1;
		else if (nItem == 5) { m_nFlags ^= 0x2; if (m_nInfoId == 4) m_nLastInfo = -1; }
		else if (nItem == 6) m_nFlags ^= 0x4;
		else if (nItem == 7) m_nFlags ^= 0x8;
		break;
	case 1:
		SetInfoId(nItem);
		break;
	case 2:
		m_nScale = nItem;
		SetSize();
		break;
	}
}

int CGDeskPluginInfoView::Activate(void)
{
	return 0;
}

int CGDeskPluginInfoView::Update(int nInactivityTicks)
{
	if (nInactivityTicks == -999)
	{
		// magic number - means 'shutting down'
		if (m_pcNotifier != NULL)
		{
			m_pcNotifier->Cancel();
			delete m_pcNotifier;
			m_pcNotifier = NULL;
		}
		return 0;
	}

	if (m_pcNotifier != NULL && m_pcNotifier->GetError())
	{
		m_nLastInfo = m_pcNotifier->GetError();
		m_pcNotifier->ClearError();
		return 1;
	}

	if (m_nInfoId == 0)
	{
		TInt nFreeRAM;
		HAL::Get(HALData::EMemoryRAMFree, nFreeRAM);
		if (nFreeRAM == m_nLastInfo) return 0;
		m_nLastInfo = nFreeRAM;
		return 1;
	}
	else if (m_nInfoId == 1)
	{
		if (m_pcNotifier == NULL) { m_nLastInfo = -2; return 1; }
		int nLevel = m_pcNotifier->GetBatteryLevel();
		if (nLevel == m_nLastInfo) return 0;
		m_nLastInfo = nLevel;
		return 1;
	}
	else if (m_nInfoId == 2)
	{
		if (m_pcNotifier == NULL) { m_nLastInfo = -2; return 1; }
		int nLevel = m_pcNotifier->GetSignalLevel();
		if (nLevel == m_nLastInfo) return 0;
		m_nLastInfo = nLevel;
		return 1;
	}
	else if (m_nInfoId == 3)
	{
		if (m_pcNotifier == NULL) { m_nLastInfo = -2; return 1; }
		int nLevel = m_pcNotifier->GetIndicators();
		if (nLevel == m_nLastInfo) return 0;
		m_nLastInfo = nLevel;
		return 1;
	}
	else if (m_nInfoId == 4)
	{
		RASCliSession cSession;
		if (cSession.Connect() != KErrNone) return 0;
		TAlarmId cId;
		int nInfo = -100;
		int nError = cSession.GetNextDueAlarmId(cId);
		if (nError == KErrNone)
		{
			TASShdAlarm cDetails;
			nError = cSession.GetAlarmDetails(cId, cDetails);
			//if (nError == KErrNone)
			{
				const TTime cTime = cDetails.NextDueTime();
				const TAlarmMessage cMsg = cDetails.Message();

				if (cTime.Int64() > 0)
				{
					TDateTime cDT = cTime.DateTime();
					nInfo = cDT.Hour() * 60 + cDT.Minute();
					if (nInfo == m_nLastInfo) { cSession.Close(); return 0; }
					if (m_nFlags & 2)
					{
						if (cMsg.Length() == 0)
							m_cInfoString.Copy(GetString(R_NO_MESSAGE));
						else
							m_cInfoString.Copy(cMsg.Left(31));
					}
					else
					{
						TTime cNow; cNow.HomeTime();
						TTimeIntervalHours cHours; cTime.HoursFrom(cNow, cHours);
						if (cHours.Int() > 24)
							cTime.FormatL(m_cInfoString, _L("%/0%*4%/1%*5 %:0%J%:1%T"));
							//cTime.FormatL(m_cInfoString, _L("%/0%*1%/1%*2%/2%*3%/3 %:0%J%:1%T"));
						else
							cTime.FormatL(m_cInfoString, _L("%:0%J%:1%T"));
					}
				}
			}
		}
		cSession.Close();
		if (nInfo == -100)
			if (m_nLastInfo == -100)
				return 0;
		m_nLastInfo = nInfo;
		return 1;
	}
	/*else if (m_nInfoId == 4)
	{
		if (m_pcNotifier == NULL) { m_nLastInfo = -2; return 1; }
		if (m_pcNotifier->GetUpdated())
		{
			m_nLastInfo = 1;
			return 1;
		}
	}*/
	return 0;
}

int CGDeskPluginInfoView::PreDraw(int nWhich, void* pvDataIn, int* pnNextTypeOut, int* pnNextParameterOut)
{
	int nRC = 0;
	switch (nWhich)
	{
	case -2:
		// if you want GDesk to allow the user to replace the visuals of this plug-in with a "Replace Icon" style graphic, return 1 for this value
		break;
	case -1:	// this case is whether we require undraw (i.e. does the draw completely cover previous draws; are there any transparent pixels)
		nRC = 1;
		break;
	case 0:
		*pnNextTypeOut = 2;	// strings list
		*pnNextParameterOut = ImplementationUID; // this value defines the name of the file to look for: in this case \Shared\GDesk\A0002187.strings
		nRC = 1;
		break;
	case 1:
		m_ppcStrings = (HBufC**)pvDataIn;	// store the strings pointer for use with GetString
		*pnNextTypeOut = 1;	// font
		*pnNextParameterOut = 9 + m_nScale * 2; // size
		nRC = 1;
		break;
	case 2:
		m_pcFont = (CFbsFont*)pvDataIn;
		break;
	}
	return nRC;
}

int CGDeskPluginInfoView::Control(int nMX, int nMY, int nMButton, int nKeyX, int nKeyY, int nKeyZ, int nKeyFire)
{
	return 0;
}

#define DrawShadowedText(_gc, _cString, _cRect, _nY, _cJustify, _nX, _bShadow, _nShadowRgb, _nTextRgb) \
	if (_bShadow) \
	{ \
		_gc.SetPenColor((TRgb)(_nShadowRgb)); \
		_gc.DrawText(_cString, _cRect, (_nY)-1, _cJustify, (_nX)); \
		_gc.DrawText(_cString, _cRect, (_nY)+1, _cJustify, (_nX)); \
		_gc.DrawText(_cString, _cRect, (_nY), _cJustify, (_nX)-1); \
		_gc.DrawText(_cString, _cRect, (_nY), _cJustify, (_nX)+1); \
	} \
	_gc.SetPenColor((TRgb)(_nTextRgb)); \
	_gc.DrawText(_cString, _cRect, (_nY), _cJustify, (_nX))

void CGDeskPluginInfoView::Draw(const TPoint& cPoint, CFbsBitGc& gc, int nHighlightRgb)
{
	TRect sRectDst(cPoint-TSize(m_cSize.iWidth/2,m_cSize.iHeight/2), m_cSize);

	if (nHighlightRgb != 0)
	{
		sRectDst.Grow(1,1);
		TRgb nClr(nHighlightRgb);
		gc.SetBrushColor(nClr); gc.SetPenColor(nClr);
		gc.SetBrushStyle(gc.ESolidBrush);
		gc.DrawRoundRect(sRectDst, TSize(6,6));
		sRectDst.Shrink(1,1);
	}

	int nFont = 9 + m_nScale * 2;
	gc.UseFont((CFont*)m_pcFont);
	gc.SetBrushStyle(gc.ENullBrush);
	TBuf<64> cString;
	if (m_nInfoId == 0)
	{
		if (m_nFlags & 2)
			cString.Format(_L("%1.1fMb"), (float)m_nLastInfo/(1024.0f*1024.0f));
		else
			cString.Format(_L("%dkb"), m_nLastInfo/1024);
	}
	else if (m_nInfoId == 1)
	{
		if (0 && m_nLastInfo < 0) cString.Format(_L("%d"), m_nLastInfo);
		else if (m_nLastInfo == -1) cString.Format(_L("-"));
		else cString.Format(_L("%d%%"), m_nLastInfo);
	}
	else if (m_nInfoId == 2)
	{
		if (0 && m_nLastInfo < 0) cString.Format(_L("%d"), m_nLastInfo);
		else if (m_nLastInfo == -1) cString.Format(_L("-"));
		//else cString.Format(_L("%d[%d]"), m_nLastInfo&15, m_nLastInfo>>4);
		else cString.Format(_L("%d"), m_nLastInfo&15);
	}
	else if (m_nInfoId == 3)
	{
		if (0 && m_nLastInfo < 0) cString.Format(_L("%d"), m_nLastInfo);
		else
		{
			int nInfo = m_nLastInfo; if (nInfo == -1) nInfo = 0;
			cString.Format(_L(""));
			if (!(m_nFlags & 2))
			{
				if (nInfo & CTelephony::KIndChargerConnected) cString.Append(GetString(R_C)); else cString.Append(_L("-"));
			}
			if (!(m_nFlags & 4))
			{
				if (nInfo & CTelephony::KIndNetworkAvailable) cString.Append(GetString(R_N)); else cString.Append(_L("-"));
			}
			if (!(m_nFlags & 8))
			{
				if (nInfo & CTelephony::KIndCallInProgress) cString.Append(GetString(R_P)); else cString.Append(_L("-"));
			}
			//cString.Format(_L("%d"), m_nLastInfo);
		}
	}
	else if (m_nInfoId == 4)
	{
		if (m_nLastInfo < 0) cString.Format(_L("-"));
		else cString.Copy(m_cInfoString);
	}
	/*else if (m_nInfoId == 4)
	{
		if (m_nLastInfo < 0) cString.Format(_L("%d"), m_nLastInfo);
		else if (m_nLastInfo < 0) cString.Format(_L("-"));
		else cString.Copy(m_pcNotifier->GetNetworkName());
	}*/

	int nLength = m_pcFont->MeasureText(cString);
	CGraphicsContext::TTextAlign nJustify;
	if (nLength < sRectDst.Width()) nJustify = CGraphicsContext::ECenter;
	else nJustify = CGraphicsContext::ELeft;
	DrawShadowedText(gc, cString, sRectDst, nFont, nJustify, 0, !(m_nFlags & 0x1), m_nBackgroundRgb.Value(), m_nTextRgb.Value());
}

const TImplementationProxy ImplementationTable[] =
{
	IMPLEMENTATION_PROXY_ENTRY(ImplementationUID,  CGDeskPluginInfoView::NewL),
};


EXPORT_C const TImplementationProxy* ImplementationGroupProxy(TInt& aTableCount)
{
	aTableCount = sizeof(ImplementationTable) / sizeof(TImplementationProxy);
	return ImplementationTable;
}
