////////////////////////////////////////
// GDesk Plug-in Interface description - part of the GDesk suite
//
// Copyright (C) 2007-8 Gary Leach
//

// Don't make changes to this file - GDesk won't understand them!

#ifndef _CGDeskInterface__
#define _CGDeskInterface__

#include <e32base.h>
#include <ECom.h>
#include <badesca.h>

#include <EIKENV.H>
#include <BitStd.h>

// UID of this interface
const TUid KCGDeskInterfaceUid = { 0xA0002181 };

//////////////////////////////////////////////////////////////////////

#ifdef S60
// Cross-platform app launcher - link against apgrfx.lib and apparc.lib
#include <APGCLI.H>
inline void StartApp(TUid cUid)
{
	RApaLsSession cLs; cLs.Connect();
	TApaAppInfo cInfo;
	cLs.GetAppInfo(cInfo, cUid);
	CApaCommandLine* cmdLine = CApaCommandLine::NewLC();
	cmdLine->SetExecutableNameL(cInfo.iFullName);
	int nErr;
	nErr = cLs.StartApp(*cmdLine);
	CleanupStack::PopAndDestroy(cmdLine); 
	cLs.Close();
}
#else
// UIQ 3 specific app launcher - precludes using plug-in on S60 - link against qikUtils.lib
#include <QikUtils.h>
inline void StartApp(TUid cUid)
{
	int nErr;
	TRAP(nErr,QikFileUtils::StartAppL(cUid));
}
#endif

//////////////////////////////////////////////////////////////////////

// flags for menu items returned using GetOptionsMenuItem
enum {
	en_MIFChecked = 0x00000001,	// item has a tick next to it
	en_MIFCascade = 0x00000002,	// item is a cascading menu item
	en_MIFSeparate = 0x00000004,	// item has a separator before it
	en_MIFGreyed = 0x00000008,		// item is greyed out
	en_MIFTypeNormal = 0x00000000,	// none of the above
	en_MIFTypeRGB = 0x00000010,	// item has an RGB colour indicator next to it (data mask defines the current RGB)
	en_MIFTypeInt = 0x00000020,	// item has an integer input mechanism (not yet supported)
	en_MIFTypeApp = 0x00000030,	// item selects an application (brings up a list of applications)
	en_MIFTypeURL = 0x00000040,	// item selects a URL (brings up a URL entry box)
	en_MIFTypeMask = 0x000000F0,
	en_MIFDataMask = 0xFFFFFF00,
};

//////////////////////////////////////////////////////////////////////

class CGDeskInterface : public CBase
{
public:
	static const int c_nInterfaceVersion = 0x00010003;

public:
	// The interface for passing initialisation parameters to the derived class constructor.
	struct TGDeskInterfaceInitParams
	{
		TInt integer;
		const TDesC* descriptor;
	};

	// Instantiates an object of this type using the aMatchString as the resolver parameters.
	static CGDeskInterface* NewL(const TDesC8& aMatchString);
	static CGDeskInterface* NewL(const TDesC8& aMatchString, TGDeskInterfaceInitParams& aParams);

	// Destructor.
	virtual ~CGDeskInterface();

	// Request a list of all available implementations which satisfy this given interface.
	static void ListAllImplementationsL(RImplInfoPtrArray& aImplInfoArray);

	virtual int InterfaceVersion(void) = 0;
	virtual int PluginVersion(void) = 0;

	///////////////////////////////////////
	// User implemented member functions

	// Control receives control information from GDesk to allow the user to interact with the plug-in in complex ways - return 1 to tell GDesk to redraw the plug-in.  Can be ignored
	// nMX, nMY are x and y positions of pen input relative to the centre of the plug-in, nMButton is 0 (no pen), 1 (pen down), 2 (pen dragging), 3 (pen up)
	// nKeyX, nKeyY and nKeyZ are joystick type inputs (nZ is -1 for keypad 1/3 and +1 for keypad 7/9)
	// nKeyFire is a select button press
	virtual int Control(int /*nMX*/, int /*nMY*/, int /*nMButton*/, int /*nKeyX*/, int /*nKeyY*/, int /*nKeyZ*/, int /*nKeyFire*/) { return 0; }
	// Not currently used
	virtual int Register(const char* pbKeyData, const TBuf16<30>& cIMEI) = 0;
	// Activate is called when the user taps on the plug-in; this can be used to launch an external app, for example
	virtual int Activate(void) = 0;
	// Update is called once per second.  nInactivityTicks gives the number of seconds since the last user input so that the plug-in can go into power-save mode.  Return 1 to tell GDesk that a redraw is required
	// GDesk will call Update with nInactivityTicks = -999 to tell the plug-in to prepare to be shut down.
	virtual int Update(int nInactivityTicks) = 0;
	// PreDraw is called multiple times to gather and request information.
	// GDesk will call PreDraw with nWhich = -2 to determine whether the visual representation of the plug-in can be replaced with by a user icon (in which case Draw never gets called)
	// GDesk will call PreDraw with nWhich = -1 to ask whether the plug-in requires an undraw before a redraw.
	// Return 0 if the plug-in completely and opaquely fills every pixel in its area (the area defined by Size() below)
	// GDesk will then call PreDraw with successive values starting at 0 to allow the plug-in to request information used for drawing.
	// Return 0 when no further information is required.
	// Return 1 and fill in pnNextTypeOut and pnNextParameterOut to get more information.  pvDataIn will contain the result on the next call of PreDraw.
	// pnNextTypeOut values:
	//  1 - request font pointer; pnNextParameterOut defines the size of the font requested in pixels
	//  2 - request string file pointer; pnNextParameterOut gives the value defining the strings (usually the UID of the plug-in); next call pvDataIn will be 
	//       an array of pointers; the first (int pointer) element in the array is the UID, the second (int pointer) is the number of strings, the remainder are HBufC* string pointers 
	//       In the plug-in PKG file make sure the string file (named <UID>.strings) is installed to /shared/GDesk/
	virtual int PreDraw(int nWhich, void* pvDataIn, int* pnNextTypeOut, int* pnNextParameterOut) = 0;
	// Draw is called when GDesk requires the plug-in to draw itself.  The centre of the draw region is defined by cPoint, gc defines the graphics context to draw to and 
	// nHighlightRGB defines the RGB colour that the plug-in should be highlighted in.
	virtual void Draw(const TPoint& cPoint, CFbsBitGc& gc, int nHighlightRGB) = 0;
	// Size returns the current size in pixels of the plug-in.  GDesk stores the position of the plug-in itself and 
	virtual const TSize& Size(void) = 0;
	// GetOptionsMenuItem is called when GDesk wants information on menu items to display.
	// nIndex starts at 0 and continues incrementing until 0 is returned.
	// If the user selects a cascading menu item then this function is called again starting at (nIndex of parent+1)*100+0
	// To add another menu item set cString to be the text, nFlags to be the menu flags (see enum above) and return 1
	virtual int GetOptionsMenuItem(int nIndex, TPtrC& cString, int& nFlags) = 0;
	// SelectOptionsMenuItem is called by GDesk to tell the plug-in that the user has selected an option.  The plug-in should update its internal information ready for re-display.
	// nIndex defines the menu item selected; nValue allows an optional value to be specified, used when the menu item type is RGB or application
	virtual void SelectOptionsMenuItem(int nIndex, int nValue = 0) = 0;
	// SetOptionsMenuItemString and getOptionsMenuItemString are used to get and set the string associated with a particular menu item index.
	// These are mainly used for things like URL entry - on starting the URL entry GDesk will call Get and if the user confirms a new URL GDesk will call Set
	virtual void SetOptionsMenuItemString(int nIndex, TPtrC& cString) { (void)nIndex; (void)cString; }
	virtual void GetOptionsMenuItemString(int nIndex, TPtrC& cString) { (void)nIndex; (void)cString; }

	// GetWriteSize is called by GDesk to determine the size of options data.  A block of data of that size will then be passed to Write and Read
	virtual int GetWriteSize(void) = 0;
	// Write is called when GDesk is about to save current settings.  The plug-in should fill in the block of data with all data that requires saving
	virtual void Write(char* pbData) = 0;
	// Read is called when GDesk is reading settings.  The plug-in should read the information and set its state accordingly.
	// nBytes is passed in so that new versions of a plug-in can maintain backward-compatibility by handling missing data.
	virtual void Read(char* pbData, int nBytes) = 0;

protected:
	//Default c'tor
	inline CGDeskInterface();

private:
	// Unique instance identifier key
	TUid iDtor_ID_Key;
};

inline CGDeskInterface::CGDeskInterface()
{
}

inline CGDeskInterface::~CGDeskInterface()
{
	// Destroy any instance variables and then inform the framework that this specific instance of the interface has been destroyed.
	REComSession::DestroyedImplementation(iDtor_ID_Key);
}

inline CGDeskInterface* CGDeskInterface::NewL(const TDesC8& aMatchString)
{
	// Set up the interface find for the default resolver.
	TEComResolverParams resolverParams;
	resolverParams.SetDataType(aMatchString);
	resolverParams.SetWildcardMatch(ETrue);		// Allow wildcard matching

	// Set up some empty initialisation parameters
	TGDeskInterfaceInitParams initParams;
	initParams.integer		= 0;
	initParams.descriptor	= NULL;

    TAny* ptr = REComSession::CreateImplementationL(
        KCGDeskInterfaceUid,
        _FOFF(CGDeskInterface,iDtor_ID_Key),
        resolverParams);

    return REINTERPRET_CAST(CGDeskInterface*, ptr);
	//return REINTERPRET_CAST(CGDeskInterface*, 
	//	REComSession::CreateImplementationL(KCGDeskInterfaceUid, _FOFF(CGDeskInterface,iDtor_ID_Key)));//, &initParams));//, resolverParams));
}

inline CGDeskInterface* CGDeskInterface::NewL(const TDesC8& aMatchString, TGDeskInterfaceInitParams& aParams)
{
	(void)aParams;

	// Set up the interface find for the default resolver.
	TEComResolverParams resolverParams;
	resolverParams.SetDataType(aMatchString);
	resolverParams.SetWildcardMatch(ETrue);		// Allow wildcard matching

	// The CreateImplementationL method will return
	// the created item.
	return REINTERPRET_CAST(CGDeskInterface*, 
		REComSession::CreateImplementationL(KCGDeskInterfaceUid, _FOFF(CGDeskInterface,iDtor_ID_Key)));//, &aParams));//, resolverParams));
}

inline void CGDeskInterface::ListAllImplementationsL(RImplInfoPtrArray& aImplInfoArray)
{
	int i;
	for (i = 0; i < 20; i ++)
	{
		int nRC;
		TRAP(nRC, REComSession::ListImplementationsL(KCGDeskInterfaceUid, aImplInfoArray));
		if (nRC == KErrNone) return;
		if (nRC == KErrNotFound || nRC == KErrArgument) return; // real error!
		// otherwise KErrNotConnected  (which doesn't exist in the SDK)
		// wait for a bit and try again
		User::After(100);
	}
}

#endif 

